<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists( 'WooCommerce' )) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Background;

/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class TemplateMelaCore_Elementor_Widget_Products extends \Elementor\Widget_Base {


    public function get_categories() {
        return array('tmcore-addons');
    }

    /**
     * Get widget name.
     *
     * Retrieve tabs widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'tmcore-products';
    }

    /**
     * Get widget title.
     *
     * Retrieve tabs widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return esc_html__('TM Products', 'templatemela-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve tabs widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-tabs';
    }


    public function get_script_depends() {
        return [
            'tmcore-elementor-products',
            'base-splide',
        ];
    }

    /**
     * Register tabs widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {

        //Section Query
        $this->start_controls_section(
            'section_setting',
            [
                'label' => esc_html__('Settings', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $options_product_type = [
            'newest'       => esc_html__('Newest Products', 'templatemela-core'),
            'on_sale'      => esc_html__('On Sale Products', 'templatemela-core'),
            'best_selling' => esc_html__('Best Selling', 'templatemela-core'),
            'top_rated'    => esc_html__('Top Rated', 'templatemela-core'),
            'featured'     => esc_html__('Featured Product', 'templatemela-core'),
        ];

        if (function_exists( 'elementor_pro_load_plugin' )) {
            $options_product_type['ids'] = esc_html__('Product Ids', 'templatemela-core');
        }

        $this->add_control(
            'product_type',
            [
                'label'   => esc_html__('Product Type', 'templatemela-core'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'newest',
                'options' => $options_product_type,
            ]
        );

        if (function_exists( 'elementor_pro_load_plugin' )) {
            $this->add_control(
                'product_ids',
                [
                    'label'        => __('Product ids', 'templatemela-core'),
                    'type'         => ElementorPro\Modules\QueryControl\Module::QUERY_CONTROL_ID,
                    'label_block'  => true,
                    'autocomplete' => [
                        'object' => ElementorPro\Modules\QueryControl\Module::QUERY_OBJECT_POST,
                        'query'  => [
                            'post_type' => 'product',
                        ],
                    ],
                    'multiple'     => true,
                    'condition'    => [
                        'product_type' => 'ids'
                    ]
                ]
            );
        }

        $this->add_control(
            'limit',
            [
                'label'   => esc_html__('Posts Per Page', 'templatemela-core'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 6,
            ]
        );

        $this->add_responsive_control(
            'column',
            [
                'label'          => esc_html__('Columns', 'templatemela-core'),
                'type'           => \Elementor\Controls_Manager::SELECT,
                'default'        => 6,
                'laptop_default' => 4,
                'tablet_default' => 3,
                'mobile_default' => 2,
                'options'        => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6, 7 => 7, 8 => 8],
            ]
        );


        $this->add_control(
            'advanced',
            [
                'label'     => esc_html__('Advanced', 'templatemela-core'),
                'type'      => Controls_Manager::HEADING,
                'condition' => [
                    'product_type!' => 'ids'
                ]
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label'     => esc_html__('Order By', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'date',
                'options'   => [
                    'date'       => esc_html__('Date', 'templatemela-core'),
                    'id'         => esc_html__('Post ID', 'templatemela-core'),
                    'menu_order' => esc_html__('Menu Order', 'templatemela-core'),
                    'popularity' => esc_html__('Number of purchases', 'templatemela-core'),
                    'rating'     => esc_html__('Average Product Rating', 'templatemela-core'),
                    'title'      => esc_html__('Product Title', 'templatemela-core'),
                    'rand'       => esc_html__('Random', 'templatemela-core'),
                ],
                'condition' => [
                    'product_type!' => 'ids'
                ]
            ]
        );

        $this->add_control(
            'order',
            [
                'label'     => esc_html__('Order', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'desc',
                'options'   => [
                    'asc'  => esc_html__('ASC', 'templatemela-core'),
                    'desc' => esc_html__('DESC', 'templatemela-core'),
                ],
                'condition' => [
                    'product_type!' => 'ids'
                ]
            ]
        );

        $this->add_control(
            'categories',
            [
                'label'       => esc_html__('Categories', 'templatemela-core'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_product_categories(),
                'label_block' => true,
                'multiple'    => true,
                'condition'   => [
                    'product_type!' => 'ids'
                ]
            ]
        );

        $this->add_control(
            'cat_operator',
            [
                'label'     => esc_html__('Category Operator', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'IN',
                'options'   => [
                    'AND'    => esc_html__('AND', 'templatemela-core'),
                    'IN'     => esc_html__('IN', 'templatemela-core'),
                    'NOT IN' => esc_html__('NOT IN', 'templatemela-core'),
                ],
                'condition' => [
                    'categories!' => ''
                ],
            ]
        );

        $this->add_control(
            'tag',
            [
                'label'       => esc_html__('Tags', 'templatemela-core'),
                'type'        => Controls_Manager::SELECT2,
                'label_block' => true,
                'options'     => $this->get_product_tags(),
                'multiple'    => true,
                'condition'   => [
                    'product_type!' => 'ids'
                ]
            ]
        );

        $this->add_control(
            'tag_operator',
            [
                'label'     => esc_html__('Tag Operator', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'IN',
                'options'   => [
                    'AND'    => esc_html__('AND', 'templatemela-core'),
                    'IN'     => esc_html__('IN', 'templatemela-core'),
                    'NOT IN' => esc_html__('NOT IN', 'templatemela-core'),
                ],
                'condition' => [
                    'tag!' => ''
                ],
            ]
        );

        $this->add_control(
            'paginate',
            [
                'label'   => esc_html__('Paginate', 'templatemela-core'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'none',
                'options' => [
                    'none'       => esc_html__('None', 'templatemela-core'),
                    'pagination' => esc_html__('Pagination', 'templatemela-core'),
                ],
            ]
        );

        $this->add_control(
            'show_sale_countdown',
            [
                'label' => esc_html__('Show Sale Countdown?', 'templatemela-core'),
                'type'  => Controls_Manager::SWITCHER,
                /* 'condition' => [
                    'product_type' => 'on_sale'
                ] */
            ]
        );

        $this->add_control(
            'product_layout',
            [
                'label'   => esc_html__('Product Layout', 'templatemela-core'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'grid',
                'options' => [
                    'grid' => esc_html__('Grid', 'templatemela-core'),
                    'list' => esc_html__('List', 'templatemela-core'),
                ],
            ]
        );

        $this->add_control(
            'list_layout',
            [
                'label'     => esc_html__('List Layout', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => '1',
                'options'   => [
                    '1' => esc_html__('Style 1', 'templatemela-core'),
                    '2' => esc_html__('Style 2', 'templatemela-core'),
                    '3' => esc_html__('Style 3', 'templatemela-core'),
                    '4' => esc_html__('Style 4', 'templatemela-core'),
                    '5' => esc_html__('Style 5', 'templatemela-core'),
                ],
                'condition' => [
                    'product_layout' => 'list'
                ]
            ]
        );

        $this->add_responsive_control(
            'list_image_size',
            [
                'label' => esc_html__( 'Image Width', 'templatemela-core' ) . ' (%)',
                'type'       => Controls_Manager::SLIDER,
                'default' => [
					'size' => 30,
					'unit' => '%',
				],
				'laptop_default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
                'selectors'  => [
                    '{{WRAPPER}} .products-list img'      =>  'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'product_layout' => 'list'
                ]
            ]
        );

        $this->add_control(
            'hide_except',
            [
                'label'      => esc_html__('Hide Except', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '3',
                                ],
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '5',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $this->add_control(
            'hide_category',
            [
                'label'      => esc_html__('Hide Category', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '2',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '3',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '5',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $this->add_control(
            'hide_button',
            [
                'label'      => esc_html__('Hide Button', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '2',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '4',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $this->add_control(
            'hide_rating',
            [
                'label'      => esc_html__('Hide Rating', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '4',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '5',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $this->add_responsive_control(
			'product_gutter',
			[
				'label'      => __( 'Grid Gutter', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'default'    =>  [
					'size' => 20,
					'unit' => 'px',
				],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} ul.products li.product'      => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} ul.products li.product-item' => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: calc({{SIZE}}{{UNIT}} - 1px);',
					'{{WRAPPER}} .products-list ul.products > li'            => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: calc({{SIZE}}{{UNIT}} - 1px);',
					'{{WRAPPER}} ul.products'                 => 'margin-left: calc({{SIZE}}{{UNIT}} / -2); margin-right: calc({{SIZE}}{{UNIT}} / -2);',
				],
                'condition'   => [
                    'enable_carousel!' => 'yes'
                ]
			]
		);

        $this->end_controls_section();
        // End Section Query

        // Carousel Option
        $this->add_control_carousel();
    }

    protected function get_product_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->slug] = $category->name;
            }
        }

        return $results;
    }

    protected function get_product_tags() {
        $tags    = get_terms(array(
                'taxonomy'   => 'product_tag',
                'hide_empty' => false,
            )
        );
        $results = array();
        if (!is_wp_error($tags)) {
            foreach ($tags as $tag) {
                $results[$tag->slug] = $tag->name;
            }
        }

        return $results;
    }

    protected function get_product_type($atts, $product_type) {
        switch ($product_type) {
            case 'featured':
                $atts['visibility'] = "featured";
                break;

            case 'on_sale':
                $atts['on_sale'] = true;
                break;

            case 'best_selling':
                $atts['best_selling'] = true;
                break;

            case 'top_rated':
                $atts['top_rated'] = true;
                break;

            default:
                break;
        }

        return $atts;
    }

    /**
     * Render tabs widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        $this->woocommerce_default($settings);
    }

    private function woocommerce_default($settings) {
        $type = 'products';
        $atts = [
            'limit'          => $settings['limit'],
            'columns'        => isset($settings['column']) ? $settings['column'] : 6 ,
            'column_laptop'  => isset($settings['column_laptop']) ? $settings['column_laptop'] : 4 ,
            'column_tablet'  => isset($settings['column_tablet']) ? $settings['column_tablet'] : 3 ,
            'column_mobile'  => isset($settings['column_mobile']) ? $settings['column_mobile'] : 2 ,
            'orderby'        => $settings['orderby'],
            'order'          => $settings['order'],
            'product_layout' => $settings['product_layout'],
        ];

        $class = '';


        if ($settings['product_layout'] == 'list') {

            $class .= ' products-list';
            $class .= ' products-list-' . $settings['list_layout'];
            $class .= ' woocommerce-product-list';

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '2') {
                $atts['show_category'] = true;
                $atts['show_button']   = true;
            }

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '3') {
                $atts['show_category'] = true;
                $atts['show_except']   = true;
            }

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '4') {
                $atts['show_rating']    = true;
                $atts['show_button']   = true;
            }

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '5') {
                $atts['show_category']  = true;
                $atts['show_except']    = true;
                $atts['show_rating']    = true;
            }

            if ($settings['hide_except'] == 'yes') {
                $atts['show_except'] = '';
            }

            if ($settings['hide_category'] == 'yes') {
                $atts['show_category'] = '';
            }

            if ($settings['hide_button'] == 'yes') {
                $atts['show_button'] = '';
            }

            if ($settings['hide_rating'] == 'yes') {
                $atts['show_rating'] = '';
            }

        }
        if (!empty($settings['show_sale_countdown']) && $settings['show_sale_countdown'] == 'yes') {
            $atts['show_countdown'] = true;   
        }
        

        $atts = $this->get_product_type($atts, $settings['product_type']);
        if (isset($atts['on_sale']) && wc_string_to_bool($atts['on_sale'])) {
            $type = 'sale_products';
        } elseif (isset($atts['best_selling']) && wc_string_to_bool($atts['best_selling'])) {
            $type = 'best_selling_products';
        } elseif (isset($atts['top_rated']) && wc_string_to_bool($atts['top_rated'])) {
            $type = 'top_rated_products';
        }

        if (isset($settings['product_ids']) && !empty($settings['product_ids']) && $settings['product_type'] == 'ids') {
            $atts['ids'] = implode(',', $settings['product_ids']);
        }

        if (!empty($settings['categories'])) {
            $atts['category']     = implode(',', $settings['categories']);
            $atts['cat_operator'] = $settings['cat_operator'];
        }

        if (!empty($settings['tag'])) {
            $atts['tag']          = implode(',', $settings['tag']);
            $atts['tag_operator'] = $settings['tag_operator'];
        }

        // Carousel
        if ($settings['enable_carousel'] === 'yes') {
            $atts['carousel_settings'] = json_encode(wp_slash($this->get_carousel_settings()));
            $atts['product_layout']    = 'carousel';
            if ($settings['product_layout'] == 'list') {
                $atts['product_layout'] = 'list-carousel';
            }
        } else {
            if (!empty($atts['column_laptop'])) {
                $class .= ' columns-laptop-' . $atts['column_laptop'];
            }
            if (!empty($atts['column_tablet'])) {
                $class .= ' columns-tablet-' . $atts['column_tablet'];
            }
            if (!empty($atts['column_mobile'])) {
                $class .= ' columns-mobile-' . $atts['column_mobile'];
            }
        }
        $atts['class'] = $class;

        if ($settings['paginate'] === 'pagination') {
            $atts['paginate'] = 'true';
        }

        $atts['cache'] = false; // disable transients for cache

        echo (new WC_Shortcode_Products($atts, $type))->get_content(); // WPCS: XSS ok
    }

    protected function add_control_carousel($condition = array()) {
        $this->start_controls_section(
            'section_carousel_options',
            [
                'label'     => esc_html__('Carousel Options', 'templatemela-core'),
                'type'      => Controls_Manager::SECTION,
                'condition' => $condition,
            ]
        );

        $this->add_control(
            'enable_carousel',
            [
                'label' => esc_html__('Enable', 'templatemela-core'),
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'dimensions',
            [
                'label'          => esc_html__('Dimensions', 'templatemela-core'),
                'type'           => \Elementor\Controls_Manager::SELECT,
                'default'        => 1,
                'options'        => [1 => 1, 2 => 2, 3 => 3],
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_responsive_control(
            'gap',
            [
                'label'   => esc_html__('Slide Gap', 'templatemela-core'),
                'type'    => Controls_Manager::NUMBER,
                'default'        => 20,
                'laptop_default' => 20,
                'tablet_default' => 15,
                'mobile_default' => 10,
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'navigation',
            [
                'label'     => esc_html__('Navigation', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'arrows',
                'options'   => [
                    'both'   => esc_html__('Arrows and Dots', 'templatemela-core'),
                    'arrows' => esc_html__('Arrows', 'templatemela-core'),
                    'dots'   => esc_html__('Dots', 'templatemela-core'),
                    'none'   => esc_html__('None', 'templatemela-core'),
                ],
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        // add dots bg color
        $this->add_control(
            'dots_bg_color',
            [
                'label'     => esc_html__('Dots Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .splide .splide__pagination__page' => 'background: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'arrows',
                        ],
                ],
                ],
            ]
        );

        $this->add_control(
            'dots_active_bg_color',
            [
                'label'     => esc_html__('Active Dots Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .splide .splide__pagination__page.is-active' => 'background: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'arrows',
                        ],
                ],
                ],
            ]
        );

        $this->add_control(
            'pause_on_hover',
            [
                'label'     => esc_html__('Pause on Hover', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label'     => esc_html__('Autoplay', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'no',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label'     => esc_html__('Autoplay Speed', 'templatemela-core'),
                'type'      => Controls_Manager::NUMBER,
                'default'   => 3000,
                'condition' => [
                    'autoplay'        => 'yes',
                    'enable_carousel' => 'yes'
                ],
                'frontend_available' => true,
            ]
        );

        $this->add_control(
            'infinite',
            [
                'label'     => esc_html__('Infinite Loop', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'no',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'carousel_arrows',
            [
                'label'      => esc_html__('Carousel Arrows', 'templatemela-core'),
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'dots',
                        ],
                    ],
                ],
            ]
        );

        $this->add_control(
            'display_on_hover',
            [
                'label'     => esc_html__('Disaply on Hover', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow' => 'opacity: 0;',
                    '{{WRAPPER}} .splide:hover .splide__arrow' => 'opacity: 1;',
                ],
            ]
        );
       
        $this->add_control(
            'next_heading',
            [
                'label' => esc_html__('Next button', 'templatemela-core'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'next_vertical',
            [
                'label'       => esc_html__('Next Vertical', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'     => 'top'
            ]
        );

        $this->add_responsive_control(
            'next_vertical_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => '%',
                    'size' => 35,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--next' => 'top: unset; bottom: unset; {{next_vertical.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );
        $this->add_control(
            'next_horizontal',
            [
                'label'       => esc_html__('Next Horizontal', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'     => 'right'
            ]
        );
        $this->add_responsive_control(
            'next_horizontal_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 0,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--next' => 'left: unset; right: unset;{{next_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );


        $this->add_control(
            'prev_heading',
            [
                'label'     => esc_html__('Prev button', 'templatemela-core'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'prev_vertical',
            [
                'label'       => esc_html__('Prev Vertical', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'     => 'top'
            ]
        );

        $this->add_responsive_control(
            'prev_vertical_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => '%',
                    'size' => 35,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--prev' => 'top: unset; bottom: unset; {{prev_vertical.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );
        $this->add_control(
            'prev_horizontal',
            [
                'label'       => esc_html__('Prev Horizontal', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'     => 'left'
            ]
        );
        $this->add_responsive_control(
            'prev_horizontal_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 0,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--prev' => 'left: unset; right: unset; {{prev_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );


        $this->end_controls_section();
		
	    // LIST STYLE
		$this->start_controls_section(
			'section_style_content',
			[
				'label' => esc_html__( 'List', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_control(
            'list_product_bg',
            [
                'label'     => __('Background Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list ul.products > li' => 'background: {{VALUE}};',
                ],
            ]
        );
		
		$this->add_responsive_control(
			'list_product_padding',
			[
				'label' => esc_html__( 'Padding', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .products-list ul.products > li' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		
		$this->add_responsive_control(
			'text_align',
			[
				'label' => esc_html__( 'Alignment', 'templatemela-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'templatemela-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'templatemela-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'templatemela-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .products-list ul.products > li' => 'text-align: {{VALUE}};',
				],
			]
		);
		
		$this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'list_product_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .products-list ul.products > li',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'list_product_radius',
            [
                'label'      => __('Border Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .products-list ul.products > li' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'list_product_box_shadow',
                'selector' => '{{WRAPPER}} .products-list ul.products > li',
            ]
        );

		$this->add_control(
			'list_title_heading',
			[
				'label' => esc_html__( 'Title', 'templatemela-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tilte_typography',
                'selector' => '{{WRAPPER}} .products-list .product-title',
            ]
        );
		
		$this->add_control(
            'title_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list .product-title' => 'color: {{VALUE}};',
                ],
            ]
        );
		
		$this->add_control(
            'title_hover_color',
            [
                'label' => __('Hover Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list .product-title:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
		
		 $this->add_responsive_control(
            'title_margin',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .products-list .product-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
		
		$this->add_control(
			'list_price_heading',
			[
				'label' => esc_html__( 'Price', 'templatemela-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_typography',
                'selector' => '{{WRAPPER}} .products-list .woocommerce-Price-amount',
            ]
        );
		
		$this->add_control(
            'price_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list .woocommerce-Price-amount' => 'color: {{VALUE}};',
                ],
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
			'section_style_product_icon',
			[
				'label' => esc_html__( 'Carousel Arrows', 'templatemela-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'dots',
                        ],
                    ],
                ],
			]
		);

        $this->start_controls_tabs( 'icon_style', [
            'condition' => $condition,
		] );

        $this->start_controls_tab(
			'icon_normal',
			[
				'label' => esc_html__( 'Normal', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'icon_color',
			[
				'label' => esc_html__( 'Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow' => 'color: {{VALUE}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'icon_bg_color',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
					'color' => [
						'global' => [
							'default' => '',
						],
					],
				],
			]
		);

		$this->end_controls_tab();

        $this->start_controls_tab(
			'icon_hover',
			[
				'label' => esc_html__( 'Hover', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'icon_hover_color',
			[
				'label' => esc_html__( 'Text Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'icon_bg_hover_color',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

        $this->add_control(
			'icon_hover_border_color',
			[
				'label' => esc_html__( 'Border Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

        $this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'icon_radius',
			[
				'label' => esc_html__( 'Border Radius', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'icon_box_shadow',
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
			]
		);

		$this->end_controls_section();
		
    }

    protected function get_carousel_settings() {
        $settings = $this->get_settings_for_display();

        return array(
            'dimensions'       => $settings['dimensions'],
            'navigation'         => $settings['navigation'],
            'autoplayHoverPause' => $settings['pause_on_hover'] === 'yes' ? true : false,
            'autoplay'           => $settings['autoplay'] === 'yes' ? true : false,
            'autoplay_speed'     => $settings['autoplay_speed'],
            'items'              => $settings['column'],
            'items_laptop'       => isset($settings['column_laptop']) ? $settings['column_laptop'] : $settings['column'],
            'items_tablet'       => isset($settings['column_tablet']) ? $settings['column_tablet'] : $settings['column'],
            'items_mobile'       => isset($settings['column_mobile']) ? $settings['column_mobile'] : 1,
            'loop'               => $settings['infinite'] === 'yes' ? true : false,
            'gap'              => $settings['gap'], 
            'gap_laptop'       => isset($settings['gap_laptop']) ? $settings['gap_laptop'] : 20,
            'gap_tablet'       => isset($settings['gap_tablet']) ? $settings['gap_tablet'] : 15,
            'gap_mobile'       => isset($settings['gap_mobile']) ? $settings['gap_mobile'] : 10,
            'product_layout'   => $settings['product_layout'],
        );
    }

    protected function render_carousel_template() {
        ?>
        var carousel_settings = {
        dimensions: settings.dimensions,
        navigation: settings.navigation,
        autoplayHoverPause: settings.pause_on_hover === 'yes' ? true : false,
        autoplay: settings.autoplay === 'yes' ? true : false,
        autoplay_speed: settings.autoplay_speed,
        items: settings.column,
        items_laptop: settings.column_laptop ? settings.column_laptop : settings.column,
        items_tablet: settings.column_tablet ? settings.column_tablet : settings.column,
        items_mobile: settings.column_mobile ? settings.column_mobile : 1,
        loop: settings.infinite === 'yes' ? true : false,
        gap: settings.gap,
        gap_laptop: settings.gap_laptop ? settings.gap_laptop : 20,
        gap_tablet: settings.gap_tablet ? settings.gap_tablet : 15,
        gap_mobile: settings.gap_mobile ? settings.gap_mobile : 10,
        product_layout: settings.product_layout,
        };
        <?php
    }
}

$widgets_manager->register(new TemplateMelaCore_Elementor_Widget_Products());
