<?php

namespace Elementor;
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
/**
 * Elementor image box widget.
 *
 * Elementor widget that displays an image, a headline and a text.
 *
 * @since 1.0.0
 */
class TMCore_Widget_Image_Box extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve image box widget name.
	 *
	 * @return string Widget name.
	 * @since  1.0.0
	 * @access public
	 *
	 */
	public function get_name() {
		return 'tmcore-image-box';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve image box widget title.
	 *
	 * @return string Widget title.
	 * @since  1.0.0
	 * @access public
	 *
	 */
	public function get_title() {
		return __( 'TM Image Box', 'templatemela-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image box widget icon.
	 *
	 * @return string Widget icon.
	 * @since  1.0.0
	 * @access public
	 *
	 */
	public function get_icon() {
		return 'eicon-image-box';
	}

	public function get_categories() {
		return array( 'tmcore-addons' );
	}

	/**
	 * Register image box widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls($condition = array()) {
		$this->start_controls_section(
			'section_image',
			[
				'label' => __( 'Image Box', 'templatemela-core' ),
			]
		);

		$this->add_control(
			'image',
			[
				'label'   => __( 'Choose Image', 'templatemela-core' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],

			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'thumbnail',
				'default'   => 'full',
				'separator' => 'none',
			]
		);


		$this->add_control(
			'title_text',
			[
				'label'       => __( 'Title & Description', 'templatemela-core' ),
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'default'     => __( 'This is the heading', 'templatemela-core' ),
				'placeholder' => __( 'Enter your title', 'templatemela-core' ),
				'label_block' => true,
			]
		);

		$this->add_control(
			'sub_title_text',
			[
				'label'       => __( 'Sub Title', 'templatemela-core' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => __( 'Enter your sub-title', 'templatemela-core' ),
				'label_block' => true,
			]
		);

		$this->add_control(
			'description_text',
			[
				'label'       => __( 'Description', 'templatemela-core' ),
				'type'        => Controls_Manager::TEXTAREA,
				'dynamic'     => [
					'active' => true,
				],
				'default'     => __( 'Click edit button to change this text. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'templatemela-core' ),
				'placeholder' => __( 'Enter your description', 'templatemela-core' ),
				'separator'   => 'none',
				'rows'        => 10,
			]
		);

		$this->add_control(
			'button_text',
			[
				'label'   => __( 'Button Text', 'templatemela-core' ),
				'type'    => Controls_Manager::TEXT,
				'default' => __( 'Click Here', 'templatemela-core' ),
			]
		);

		$this->add_control(
			'link_button',
			[
				'label'       => __( 'Link', 'templatemela-core' ),
				'type'        => Controls_Manager::URL,
				'placeholder' => __( 'https://your-link.com', 'templatemela-core' ),
				'default'     => [
					'url' => '#',
				],
			]
		);

		$this->add_control(
			'title_size',
			[
				'label'   => __( 'Title HTML Tag', 'templatemela-core' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'h1'   => 'H1',
					'h2'   => 'H2',
					'h3'   => 'H3',
					'h4'   => 'H4',
					'h5'   => 'H5',
					'h6'   => 'H6',
					'div'  => 'div',
					'span' => 'span',
					'p'    => 'p',
				],
				'default' => 'h3',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_wrap_style',
			[
				'label' => __( 'Wrap', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
            'wrap_background',
            [
                'label'     => __('Background Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .tmcore-image-box-content' => 'background: {{VALUE}};',
                ],
            ]
        );

		$this->add_responsive_control(
			'wrap_padding',
			[
				'label'      => __( 'Padding', 'templatemela-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .tmcore-image-box-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'wrap_align',
			[
				'label' => esc_html__( 'Alignment', 'templatemela-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'templatemela-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'templatemela-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'templatemela-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-content' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_control(
            'wrap_vertical_position',
            [
                'label'       => esc_html__('Vertical Position', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
				'default'     => 'top'
            ]
        );

        $this->add_responsive_control(
            'wrap_vertical_position_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 15,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .tmcore-image-box-content' => 'top: unset; bottom: unset; {{wrap_vertical_position.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

		$this->add_control(
            'wrap_horizontal_position',
            [
                'label'       => esc_html__('Horizontal Position', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'     => 'left'
            ]
        );
        $this->add_responsive_control(
            'wrap_horizontal_position_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 15,
                ],
                'selectors'  => [
					'{{WRAPPER}} .tmcore-image-box-content' => 'left: unset; right: unset; {{wrap_horizontal_position.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );


		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			[
				'label' => __( 'Title', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_ACCENT,
                ],
				'selector' => '{{WRAPPER}} .tmcore-image-box-title',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => __( 'Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
                'global' => [
                    'default' => Global_Colors::COLOR_ACCENT,
                ],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_space',
			[
				'label'      => __( 'Space', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .tmcore-image-box-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		// sub

		$this->start_controls_section(
			'section_subtitle_style',
			[
				'label' => __( 'Sub Title', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_TEXT,
                ],
				'selector' => '{{WRAPPER}} .tmcore-image-box-sub-title',
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => __( 'Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
                'global' => [
                    'default' => Global_Colors::COLOR_TEXT,
                ],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-sub-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'subtitle_space',
			[
				'label'      => __( 'Space', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .tmcore-image-box-sub-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_description_style',
			[
				'label' => __( 'Description', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'description_typography',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_TEXT,
                ],
				'selector' => '{{WRAPPER}} .tmcore-image-box-description',
			]
		);

		$this->add_control(
			'description_color',
			[
				'label'     => __( 'Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
                'global' => [
                    'default' => Global_Colors::COLOR_TEXT,
                ],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'description_space',
			[
				'label'      => __( 'Space', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .tmcore-image-box-description' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image_button',
			[
				'label' => esc_html__( 'Button', 'templatemela-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'image_button_typography',
				'global' => [
					'default' => Global_Typography::TYPOGRAPHY_ACCENT,
				],
				'selector' => '{{WRAPPER}} .tmcore-image-box-button',
			]
		);

        $this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'image_button_shadow',
				'selector' => '{{WRAPPER}} .tmcore-image-box-button',
			]
		);

        $this->start_controls_tabs( 'image_button_style', [
			'condition' => $condition,
		] );

        $this->start_controls_tab(
			'image_button_normal',
			[
				'label' => esc_html__( 'Normal', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'image_button_text_color',
			[
				'label' => esc_html__( 'Text Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-button' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'background',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .tmcore-image-box-button',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
					'color' => [
						'global' => [
							'default' => Global_Colors::COLOR_ACCENT,
						],
					],
				],
			]
		);

		$this->end_controls_tab();

        $this->start_controls_tab(
			'image_button_hover',
			[
				'label' => esc_html__( 'Hover', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'image_button_hover_color',
			[
				'label' => esc_html__( 'Text Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-button:hover, {{WRAPPER}} .tmcore-image-box-button:focus' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'image_button_background_hover',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .tmcore-image-box-button:hover, {{WRAPPER}} .tmcore-image-box-button:focus',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

        $this->add_control(
			'image_button_hover_border_color',
			[
				'label' => esc_html__( 'Border Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-button:hover, {{WRAPPER}} .tmcore-image-box-button:focus' => 'border-color: {{VALUE}};',
				],
			]
		);


		$this->end_controls_tab();

        $this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'selector' => '{{WRAPPER}} .tmcore-image-box-button',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'image_button_box_shadow',
				'selector' => '{{WRAPPER}} .tmcore-image-box-button',
			]
		);

		$this->add_responsive_control(
			'text_padding',
			[
				'label' => esc_html__( 'Padding', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .tmcore-image-box-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before',
			]
		);

		$this->end_controls_section();
	}


	protected function render() {
		$settings = $this->get_settings_for_display();

		$has_content = ! empty( $settings['title_text'] ) || ! empty( $settings['description_text'] );
		$this->add_render_attribute( 'wrapper', 'class', 'tmcore-image-box-wrapper' );
		if ( ! empty( $settings['link_button']['url'] ) ) {
			$this->add_render_attribute( 'button_text', 'href', $settings['link_button']['url'] );

			if ( ! empty( $settings['link_button']['is_external'] ) ) {
				$this->add_render_attribute( 'button_text', 'target', '_blank' );
			}
		}
		$this->add_render_attribute( 'button_text', 'class', 'button tmcore-image-box-button' );
		$this->add_render_attribute( 'image-wrapper', 'class', 'tmcore-image-box-img' );
		if ( ! empty( $settings['image']['url'] ) ) {
			$this->add_render_attribute( 'image', 'src', $settings['image']['url'] );
			$this->add_render_attribute( 'image', 'alt', Control_Media::get_image_alt( $settings['image'] ) );
			$this->add_render_attribute( 'image', 'title', Control_Media::get_image_title( $settings['image'] ) );
		}
		if ( $has_content ) {
			if ( ! empty( $settings['sub_title_text'] ) ) {
				$this->add_render_attribute( 'sub_title_text', 'class', 'tmcore-image-box-sub-title' );
			}
			if ( ! empty( $settings['title_text'] ) ) {
				$this->add_render_attribute( 'title_text', 'class', 'tmcore-image-box-title' );
				$this->add_inline_editing_attributes( 'title_text', 'none' );
			}
			if ( ! empty( $settings['description_text'] ) ) {
				$this->add_render_attribute( 'description_text', 'class', 'tmcore-image-box-description' );
				$this->add_inline_editing_attributes( 'description_text' );
			}
		}

		?>
        <div <?php echo tmcore_elementor_get_render_attribute_string( "wrapper", $this ); ?>>
            <div class="tmcore-image-framed">
                <figure <?php echo tmcore_elementor_get_render_attribute_string( "image-wrapper", $this ); ?>>
					<?php
					if ( ! empty( $settings['image']['url'] ) ) {
						echo Group_Control_Image_Size::get_attachment_image_html( $settings, 'thumbnail', 'image' );
					}
					?>
                </figure>
				<?php if ( $has_content ): ?>
                    <div class="tmcore-image-box-content">
						<?php if ( ! empty( $settings['sub_title_text'] ) ): ?>
                            <div <?php echo tmcore_elementor_get_render_attribute_string( "sub_title_text", $this ); ?>>
								<?php echo wp_kses_post( $settings["sub_title_text"] ); ?>
                            </div>
						<?php endif; ?>

						<?php if ( ! empty( $settings['title_text'] ) ): ?>
							<?php printf( '<%1$s %2$s>%3$s</%1$s>', $settings['title_size'], tmcore_elementor_get_render_attribute_string( 'title_text', $this ),  $settings['title_text'] ); ?>
						<?php endif; ?>

						<?php if ( ! empty( $settings['description_text'] ) ): ?>
							<?php printf( '<p %1$s>%2$s</p>', tmcore_elementor_get_render_attribute_string( 'description_text', $this ),  $settings['description_text'] ); ?>
						<?php endif; ?>
                        <?php if ( ! empty( $settings['link_button']['url'] ) ): ?>
                            <a <?php echo tmcore_elementor_get_render_attribute_string( 'button_text', $this ); ?>>
                                <?php echo esc_html( $settings['button_text'] ); ?>
                            </a>
                        <?php endif; ?>
                    </div>
				<?php endif; ?>
            </div>
        </div>
		<?php
	}
}

$widgets_manager->register( new TMCore_Widget_Image_Box() );
