<?php
/**
 * Displays an inactive message if the API License Key has not yet been activated
 *
 * @package TemplateMela
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class for API getting started page.
 *
 * @category class
 */
class TemplateMela_Core_API_Manager {
	/**
	 * This is where we make api calls.
	 *
	 * @var api url
	 */
	public $upgrade_url = 'https://templatemela.com/';

	/**
	 * This is fall back for where we make api calls.
	 *
	 * @var url
	 */
	private $fallback_api_url = 'https://templatemela.com/';

	/**
	 * This is the link to the account page.
	 *
	 * @var url
	 */
	private $renewal_url = 'https://templatemela.com/my-account/';
	
	/**
	 * This is where we make envato and templatemonster api calls.
	 *
	 * @var api url
	 */
	public $envamonster_api_url = 'https://api.templatemela.com/v1/';

	/**
	 * This is the link to the account page.
	 *
	 * @var url
	 */
	private $tmc_license_link;

	/**
	 * This is the current theme version.
	 *
	 * @var number
	 */
	public $version;

	/**
	 * This is the theme name.
	 *
	 * @var theme name
	 */
	private $tmc_product_name;

	/**
	 * This is the current theme data object.
	 *
	 * @var theme data
	 */
	private $my_theme;

	/**
	 * This is the data key for database.
	 *
	 * @var string
	 */
	public $tmc_data_key = 'tm_core_api_manager';

	/**
	 * This is the settings key for api key.
	 *
	 * @var string
	 */
	public $tmc_api_key = 'tm_core_api_key';

	/**
	 * This is the settings key for api email.
	 *
	 * @var string
	 */
	public $tmc_activation_email = 'activation_email';

	/**
	 * This is the product ID key.
	 *
	 * @var string
	 */
	public $tmc_product_id_key;

	/**
	 * This is the api instance key.
	 *
	 * @var string
	 */
	public $tmc_instance_key;

	/**
	 * This is the api activated key.
	 *
	 * @var string
	 */
	public $tmc_activated_key;

	/**
	 * This is the settings key for api checkbox.
	 *
	 * @var string
	 */
	public $tmc_deactivate_checkbox = 'tmc_deactivate_example_checkbox';

	/**
	 * This is the settings key for api activate tab.
	 *
	 * @var string
	 */
	public $tmc_activation_tab_key = 'tmc_api_manager_dashboard';

	/**
	 * This is the settings key for api deactive tab.
	 *
	 * @var string
	 */
	public $tmc_deactivation_tab_key = 'tmc_api_manager_dashboard_deactivation';
	/**
	 * This is the page menu title.
	 *
	 * @var string
	 */
	public $tmc_settings_menu_title;
	/**
	 * This is the page title.
	 *
	 * @var string
	 */
	public $tmc_settings_title;
	/**
	 * This is the activation title.
	 *
	 * @var string
	 */
	public $tmc_menu_tab_activation_title;
	/**
	 * This is the deactivation title.
	 *
	 * @var string
	 */
	public $tmc_menu_tab_deactivation_title;
	/**
	 * This is options array.
	 *
	 * @var array
	 */
	public $tmc_options;
	/**
	 * This is the product ID.
	 *
	 * @var string
	 */
	public $tmc_product_id;
	/**
	 * This is the instance ID.
	 *
	 * @var string
	 */
	public $tmc_instance_id;
	/**
	 * This is the site domain.
	 *
	 * @var string
	 */
	public $tmc_domain;

	/**
	 * This is if it's a multisite.
	 *
	 * @var bool
	 */
	public static $multisite = false;

	/**
	 * Instance Control.
	 *
	 * @var null
	 */
	protected static $_instance = null;

	public $plugin_slug;
	public $plugin_version;
	public $cache_key;
	public $cache_allowed;

	/**
	 * TGMPA instance.
	 *
	 * @var    object
	 */
	protected $tgmpa;

	/**
	 * Instance Control.
	 *
	 * @param string $tmc_product_id_key product ID key.
	 * @param string $tmc_instance_key product instance key.
	 * @param string $tmc_activated_key product activated key.
	 * @param string $tmc_product_id the product ID.
	 * @param string $tmc_product_name the product name.
	 */
	public static function instance( $tmc_product_id_key, $tmc_instance_key, $tmc_activated_key, $tmc_product_id, $tmc_product_name ) {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self( $tmc_product_id_key, $tmc_instance_key, $tmc_activated_key, $tmc_product_id, $tmc_product_name );
		}
		return self::$_instance;
	}
	/**
	 * Constructor function.
	 *
	 * @param string $tmc_product_id_key product ID key.
	 * @param string $tmc_instance_key product instance key.
	 * @param string $tmc_activated_key product activated key.
	 * @param string $tmc_product_id the product ID.
	 * @param string $tmc_product_name the product name.
	 */
	public function __construct( $tmc_product_id_key, $tmc_instance_key, $tmc_activated_key, $tmc_product_id, $tmc_product_name ) {
		
		// Get TGMPA.
		if ( class_exists( 'TGM_Plugin_Activation' ) ) {
			$this->tgmpa = isset( $GLOBALS['tgmpa'] ) ? $GLOBALS['tgmpa'] : TGM_Plugin_Activation::get_instance();
		}

		// Only run in the admin.
		if ( is_admin() ) {
			if ( is_multisite() ) {
				$show_local_activation = apply_filters( 'base_activation_individual_multisites', false );
				if ( $show_local_activation ) {
					self::$multisite = false;
				} else {
					self::$multisite = true;
				}
			}
			add_action( 'wp_loaded', array( $this, 'hide_inactive_notice' ) );
			add_action( 'admin_notices', array( $this, 'check_external_blocking' ) );
			add_action( 'admin_init', array( $this, 'activation' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'scripts' ) );
			add_filter( 'tgmpa_load', array( $this, 'load_tgmpa' ), 10, 1 );

			add_action( 'tmc_dashboard_contents_left', array( $this, 'render_license_form' ) );
			add_action( 'tmc_dashboard_contents_right', array( $this, 'render_information' ) );
			add_action( 'wp_ajax_base_add_elementor', array( $this, 'add_elementor_ajax_callback' ) );
			// Repeat Check license.
			add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'status_check' ) );

			// Testing Status Check // testing
			//add_action( 'admin_init', array( $this, 'status_check' ) );

			$this->my_theme = wp_get_theme(); // Get theme data.
			$this->version  = $this->my_theme->get( 'Version' );

			/**
			 * Set all data defaults here
			 */
			$this->tmc_product_name                 = apply_filters( 'base_whitelabel_theme_name', $tmc_product_name );
			$this->tmc_license_link                 = apply_filters( 'base_whitelabel_license_link', $this->renewal_url );
			$this->tmc_product_id_key               = $tmc_product_id_key;
			$this->tmc_instance_key                 = $tmc_instance_key;
			$this->tmc_activated_key                = $tmc_activated_key;
			$this->tmc_settings_menu_title          = __( 'Getting Started', 'templatemela-core' );
			$this->tmc_menu_tab_activation_title    = __( 'API License Activation', 'templatemela-core' );
			$this->tmc_menu_tab_deactivation_title  = __( 'Deactivation', 'templatemela-core' );
			$this->tmc_options                      = $this->get_setting_option( $this->tmc_data_key );
			$this->tmc_product_id                   = $tmc_product_id; // Software ID.
			$this->tmc_instance_id                  = $this->get_setting_option( $this->tmc_instance_key ); // Instance ID (unique to each blog activation).
			$this->tmc_domain                       = str_ireplace( array( 'http://', 'https://' ), '', home_url() );

			if ( empty( $this->tmc_instance_id ) ) {
				$this->update_setting_option( $this->tmc_instance_key, wp_generate_password( 12, false ) );
			}

			add_action( 'init', array( $this, 'options_update' ), 1 );
			add_action( 'admin_notices' , array( $this, 'admin_interface_notices' ) );
			
			if ( 'Activated' !== $this->get_setting_option( $this->tmc_activated_key ) ) {
				add_action( 'admin_notices', array( $this, 'inactive_notice' ) );
			}
		}
	}

	/**
	 * On options save.
	 */
	public function options_update() {
		if ( isset( $_POST['tmcore_license_form_submit'] ) ) {
			$this->license_form_submit();
		}
	}

	/**
	 * Conditionally load TGMPA
	 *
	 * @param string $status User's manage capabilities.
	 */
	public function load_tgmpa( $status ) {
		return is_admin() || current_user_can( 'install_themes' );
	}

	/**
	 * Save Add Elements to Elementor.
	 */
	public function add_elementor_ajax_callback() {
		if ( ! check_ajax_referer( 'base-ajax-verification', 'security', false ) ) {
			wp_send_json_error( __( 'Security Error, please reload the page.', 'templatemela-core' ) );
		}
		if ( class_exists( '\Elementor\Plugin' ) ) {
			$cpt_support = get_option( 'elementor_cpt_support' );
			if ( ! $cpt_support ) {
				$cpt_support = array( 'page', 'post', 'base_element' );
				update_option( 'elementor_cpt_support', $cpt_support );
			} else if ( ! in_array( 'base_element', $cpt_support ) ) {
				$cpt_support[] = 'base_element';
				update_option( 'elementor_cpt_support', $cpt_support );
			}
		}
		wp_send_json_success();
	}

	/**
	 * On options save validate the license.
	 */
	public function license_form_submit() {

		//check for de-activation.
		if ( isset( $_POST['tmcore_license_form_submit'] ) && isset( $_POST['tmcore_license_deactivate'] ) && wp_verify_nonce( $_POST['tmcore_license_nonce'], 'tmcore_license') ) {
			$this->validate_deactivation_options();
		}

		if ( isset( $_POST['tmcore_license_form_submit'] ) && isset( $_POST['tmcore_license_activate'] ) && wp_verify_nonce( $_POST['tmcore_license_nonce'], 'tmcore_license' ) ) {
			$this->validate_activation_options();
		}
	}

	/**
	 * Updates Settings.
	 *
	 * @param string $key the setting Key.
	 * @param mixed  $option the setting value.
	 */
	public function update_setting_option( $key, $option ) {
		if ( self::$multisite && is_multisite() ) {
			update_site_option( $key, $option );
		} else {
			update_option( $key, $option );
		}
	}

	/**
	 * Retrives Settings.
	 *
	 * @param string $key the setting Key.
	 * @param mixed  $default the setting default value.
	 */
	public function get_setting_option( $key, $default = null ) {
		if ( self::$multisite && is_multisite() ) {
			return get_site_option( $key, $default );
		} else {
			return get_option( $key, $default );
		}
	}

	/**
	 * Delete Settings.
	 *
	 * @param string $key the setting Key.
	 */
	public function delete_setting_option( $key ) {
		if ( self::$multisite && is_multisite() ) {
			delete_site_option( $key );
		} else {
			delete_option( $key );
		}
	}

	/**
	 * Activation function to set defaults.
	 */
	public function activation() {
		if ( false === $this->get_setting_option( $this->tmc_data_key ) || false === $this->get_setting_option( $this->tmc_instance_key ) ) {
			$global_options = array(
				$this->tmc_api_key           => '',
				$this->tmc_activation_email  => '',
			);
			$this->update_setting_option( $this->tmc_data_key, $global_options );
			$single_options = array(
				$this->tmc_product_id_key       => $this->tmc_product_id,
				$this->tmc_instance_key         => wp_generate_password( 12, false ),
				$this->tmc_deactivate_checkbox  => 'on',
				$this->tmc_activated_key        => 'Deactivated',
			);
			foreach ( $single_options as $key => $value ) {
				$this->update_setting_option( $key, $value );
			}
		}
	}

	/**
	 * Check the license status.
	 *
	 * @param string $transient_value filter to pass along.
	 */
	public function status_check( $transient_value = null ) {
		$status = get_transient( 'tmcore_tmc_api_status_check' );

		//$status = false; // Temporary for testing
		if ( false === $status ) {
			if ( $this->get_setting_option( $this->tmc_activated_key ) === 'Activated' ) {
				$data    = $this->get_setting_option( $this->tmc_data_key );
				$args = array(
					'email'         => ( isset( $data[ $this->tmc_activation_email ] ) ? $data[ $this->tmc_activation_email ] : '' ),
					'license_key'   => ( isset( $data[ $this->tmc_api_key ] ) ? $data[ $this->tmc_api_key ] : '' ),
					'version'       => TMCORE_VERSION,
					'envamonster_key'   => ( isset( $data['envamonster_key'] ) ? $data['envamonster_key'] : '' ),
				);
				if ( isset( $data['licensemela'] ) && $data['licensemela'] ) {
					$status_results = json_decode( $this->status( $args ), true );
				} else {
					$status_results = json_decode( $this->status_envamonster( $args ), true ); // Envato / Monster
				}
				if ( 'failed' === $status_results ) {
					// do nothing.
				} elseif ( isset( $status_results['activated'] ) && 'inactive' === $status_results['activated'] ) {
					$this->uninstall();
					$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
				} elseif ( isset( $status_results['status_check'] ) && 'inactive' === $status_results['status_check'] ) {
					$this->uninstall();
					$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
				} elseif ( isset( $status_results['error'] ) && ( '101' == $status_results['code'] || '104' == $status_results['code'] ) ) {
					$this->uninstall();
					$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
				}
			}
			set_transient( 'tmcore_tmc_api_status_check', 1, DAY_IN_SECONDS );
		}
		return $transient_value;
	}

	/**
	 * Uninstall the product license.
	 */
	public function uninstall() {

		$this->license_key_deactivation();
		foreach ( array(
			$this->tmc_data_key,
			$this->tmc_product_id_key,
			$this->tmc_instance_key,
			$this->tmc_deactivate_checkbox,
			$this->tmc_activated_key,
		) as $option ) {
			$this->delete_setting_option( $option );
		}
	}

	/**
	 * Deactivates the license on the API server
	 */
	public function license_key_deactivation() {
		$activation_status = $this->get_setting_option( $this->tmc_activated_key );

		$api_email = $this->tmc_options[ $this->tmc_activation_email ];
		$api_key   = $this->tmc_options[ $this->tmc_api_key ];

		$args = array(
			'email'       => $api_email,
			'license_key' => $api_key,
		);

		if ( 'Activated' === $activation_status && ! empty( $api_key ) && ! empty( $api_email ) ) {
			$this->deactivate( $args ); // reset license key activation.
		}
	}

	/**
	 * Hide Notice
	 */
	public function hide_inactive_notice() {
		if ( isset( $_GET['base-wp-activation-notice'] ) && isset( $_GET['_notice_nonce'] ) ) {
			if ( ! wp_verify_nonce( wp_unslash( sanitize_key( $_GET['_notice_nonce'] ) ), 'base_wp_hide_activation_notice' ) ) {
				wp_die( esc_html__( 'Authorization failed. Please refresh the page and try again.', 'templatemela-core' ) );
			}
			update_option( $this->tmc_product_id_key . '_activation_notice', true );
		}
	}

	/**
	 * Displays an inactive notice when the software is inactive.
	 */
	public function inactive_notice() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		if ( isset( $_GET['page'] ) && ( 'templatemela' === $_GET['page'] || 'templatemela-setup' === $_GET['page'] ) ) {
			return;
		}
		if ( get_option( $this->tmc_product_id_key . '_activation_notice' ) ) {
			return;
		}
		?>
		<div id="message" class="error" style="position: relative;">
			<p><?php /* translators: %1$s and %2$s refer to an internal link markup */ printf( __( 'TemplateMela license has not been activated! %1$sClick here%2$s to activate the license key.', 'templatemela-core' ), '<a href="' . esc_url( admin_url( 'admin.php?page=templatemela' ) ) . '">', '</a>' ); ?></p>
		</div>
		<?php
	}

	/**
	 * Check for external blocking contstant
	 */
	public function check_external_blocking() {
		// show notice if external requests are blocked through the WP_HTTP_BLOCK_EXTERNAL constant.
		if ( defined( 'WP_HTTP_BLOCK_EXTERNAL' ) && WP_HTTP_BLOCK_EXTERNAL === true ) {

			// check if our API endpoint is in the allowed hosts.
			$host = parse_url( $this->upgrade_url, PHP_URL_HOST );

			if ( ! defined( 'WP_ACCESSIBLE_HOSTS' ) || stristr( WP_ACCESSIBLE_HOSTS, $host ) === false ) {
				?>
				<div class="error">
				<p><?php /* translators: %1$s  refers to product name. %2$s refer to host information and %3$s an internal link markup */ printf( esc_html__( 'Warning! You\'re blocking external requests which means you won\'t be able to get %1$s updates. Please add %2$s to %3$s.', 'templatemela-core' ), $this->tmc_product_id, '<strong>' . $host . '</strong>', '<code>WP_ACCESSIBLE_HOSTS</code>' ); ?></p>
				</div>
				<?php
			}

		}
	}

	/**
	 * Show api result as admin notices
	 */
	public function admin_interface_notices() {
		global $tmcore_license_interface_messages;

		if ( isset( $_GET['page'] ) && 'templatemela' !== $_GET['page'] && 'templatemela-setup' !== $_GET['page'] ) {
			return;
		}

		if ( ! is_array( $tmcore_license_interface_messages ) ) {
			return;
		}

		if ( count( $tmcore_license_interface_messages ) > 0 ) {
			foreach ( $tmcore_license_interface_messages as $message ) {
				echo '<div class="' . esc_attr( $message['type'] ) . ' fade"><p>' . esc_html( $message['text'] ) . '</p></div>';
			}
		}
	}

	/**
	 * Render license form
	 */
	public function render_license_form() {
		?>
		<div class="license-section section-item">
				<?php
				if ( 'Activated' !== $this->get_setting_option( $this->tmc_activated_key ) ) {
					$this->license_form();
				} else {
					$this->license_deactivate_form();
				}
				?>
			</div>
		<?php
	}

	/**
	 * Render customized information
	 */
	public function render_information() {
		?>
		<div class="information-section section-item">
			<?php
			if ( 'Activated' !== $this->get_setting_option( $this->tmc_activated_key ) ) {
			?>	
				<h2><?php esc_html_e( 'TemplateMela license has not been activated!', 'templatemela-core' ); ?></h2>
				<div class="error inline notice-warning notice-alt" style="position: relative;">
					<p><?php esc_html_e( 'TemplateMela license has not been activated! Activate the license to enable remote updates and theme support.', 'templatemela-core' ); ?></p>
				</div>
			<?php
			} else {
			?>
				<h2><?php esc_html_e( 'Customize Your Site', 'templatemela-core' ); ?></h2>
				<p><?php esc_html_e( 'You can easily customise your site using the many available options such as color, font, spacing, and more!', 'templatemela-core' ); ?></p>
				<p><?php /* translators: %1$s and %2$s refer to an internal link markup */ printf( __( '%1$sClick here%2$s to see available options.', 'templatemela-core' ), '<a href="' . esc_url( admin_url( 'themes.php?page=avanam' ) ) . '">', '</a>' ); ?></p>
			<?php
			}
			if ( 'Activated' == $this->get_setting_option( $this->tmc_activated_key ) ) {
			
			if ( class_exists( 'TGM_Plugin_Activation' ) ) {
			
				// Are there plugins that need installing/activating?
				$plugins     = $this->get_tgmpa_plugins();
				$all_plugins = $install_plugins = $update_plugins = $activate_plugins = array();
				$count       = count( $plugins['all'] );
				$class       = $count ? null : 'no-plugins';

				foreach ( $plugins['install'] as $slug => $plugin ) {
					$install_plugins[] = $slug;
				}
				foreach ( $plugins['update'] as $slug => $plugin ) {
					$update_plugins[] = $slug;
				}
				foreach ( $plugins['activate'] as $slug => $plugin ) {
					$activate_plugins[] = $slug;
				}

				foreach ( $plugins['all'] as $slug => $plugin ) {
					if (in_array( $slug, $install_plugins)){
						$plugin['install'] = true;
					}
					if (in_array( $slug, $update_plugins)){
						$plugin['update'] = true;
					}
					if (in_array( $slug, $activate_plugins)){
						$plugin['activate'] = true;
					}
					$all_plugins[ $slug ] = $plugin;
				}
				?>
				<?php if ( $count ) : ?>
					<?php //$updates_required = array_merge($install_plugins,$update_plugins); ?>
					<?php //if ( ! empty( count( $updates_required ) ) ) : ?>
						<h2><?php esc_html_e( 'Download Plugins!', 'templatemela-core' ); ?></h2>
					<?php //endif; ?>

					<ul class="install-plugins">
					<?php if ( ! empty( $all_plugins ) ) : ?>
						<?php foreach ( $all_plugins as $slug => $plugin ) : ?>
							<?php //if ( ! empty( $plugin['install'] ) || ! empty( $plugin['update'] ) ) : ?>
							<li data-slug="<?php echo esc_attr( $slug ); ?>" class="update-message notice inline notice-warning notice-alt">
								<span><?php echo esc_html( $plugin['name'] ); ?> <span class="avaialble"><?php //_e( 'Update Available', 'templatemela-core' ); ?></span>
								<?php //if( 'revslider' == $slug ) : ?>
								<!-- <span class="badge">
									(<a href="<?php //echo $this->tgmpa->get_tgmpa_url(); ?>"><span><?php //_e( 'Update Now', 'templatemela-core' ); ?></span></a>)
								</span> -->
								<?php //else: ?>
								<span class="badge">
									(<a href="<?php echo $plugin['source'].$this->tmc_options[ $this->tmc_api_key ]; ?>"><span><?php _e( 'Download Now', 'templatemela-core' ); ?></span></a>)
								</span>
								<?php //endif; ?>
							</li>
							<?php //endif; ?>
						<?php endforeach; ?>
					<?php endif; ?>
					</ul>
				<?php endif; ?>

			<?php } 
			}
			?>
		</div>
		<?php
	}

	/**
	 * Render license activation form
	 */
	public function license_form() {
		$data = $this->get_setting_option( $this->tmc_data_key );
		$is_templatemela = ( isset( $data['licensemela'] ) && $data['licensemela'] ? true : false );
		
		if ( isset( $data[ $this->tmc_api_key ] ) && ! empty( $data[ $this->tmc_api_key ] ) ) {
			$input_string = $data[ $this->tmc_api_key ];
		} else {
			$input_string = '';
		}
		if ( isset( $data[ $this->tmc_activation_email ] ) && ! empty( $data[ $this->tmc_activation_email ] ) ) {
			$email_string = $data[ $this->tmc_activation_email ];
		} else {
			$email_string = '';
		}
		?>
		<form name="form" method="post" id="templatemela-license-activate">
			<input type="hidden" name="tmcore_license_form_submit" value="true" />
			<input type="hidden" name="tmcore_license_activate" value="true" />
			<?php wp_nonce_field( 'tmcore_license', 'tmcore_license_nonce' ); ?>
			
			<?php if ( isset( $_GET['page'] ) && 'templatemela-setup' !== $_GET['page'] ) { ?>
				<h2><?php echo __( 'TemplateMela License', 'templatemela-core' ); ?> <span class="tmc-license-status tmc-inactive"><?php echo __( 'Inactive', 'templatemela-core' ); ?></span></h2>
				<p class="activate-notice"><?php echo __( 'Enter your purchase code to enable remote updates and theme support.', 'templatemela-core' ); ?></p>
			<?php } ?>

			<?php //$is_templatemela = isset( $_POST['tm_core_api_manager[licensemela]'] ) ? true : false ; ?>
			<!-- <div class="toggle-license">
				<input onChange="document.getElementById('templatemela-license-activate').submit()" <?php //echo checked( $is_templatemela, true ); ?> type="checkbox" id="base_toggle_plateform" name="<?php //echo esc_attr( $this->tmc_data_key . '[licensemela]' ); ?>" value="on">
				<label for="base_toggle_plateform"><?php //echo __( 'Use TemplateMela.com License', 'templatemela-core' ); ?></label>
			</div> -->
			
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row"><label for="api_key"><?php echo 'Purchase Code'; ?></label></th>
						<td>
							<input id="api_key" name="<?php echo esc_attr( $this->tmc_data_key . '[' . $this->tmc_api_key . ']' ); ?>" size="25" type="text" value="<?php echo esc_attr( $input_string ); ?>" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
							<a href="https://templatemela.com/find-your-purchase-code" alt="Need Help?" target="_blank">
								<span class="hint--top" aria-label="Need Help?">
									<svg class="icon icon--help" aria-hidden="true" role="img"><use xlink:href="#icon-help" width="100%"></use></svg>
								</span>
							</a>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="activation_email"><?php echo 'License Email'; ?></label></th>
						<td>
							<input id="activation_email" name="<?php echo esc_attr( $this->tmc_data_key . '[' . $this->tmc_activation_email . ']' ); ?>" size="25" type="text" value="<?php echo esc_attr( $email_string ); ?>">
						</td>
					</tr>
				</tbody>
			</table>
			<p class="submit">
				<?php if ( isset( $_GET['page'] ) && 'templatemela-setup' === $_GET['page'] ) { ?>
				<?php } else { ?>
					<input type="submit" name="btn-submit" id="btn-submit" class="button button-large button-primary" value="<?php echo esc_attr__( 'Activate', 'templatemela-core' ); ?>">
				<?php } ?>
			</p>
		</form>
		<?php
	}

	/**
	 * Render license deactivation form
	 */
	public function license_deactivate_form() {
		$options = $this->get_setting_option( $this->tmc_data_key );
		$is_templatemela = ( isset( $options['licensemela'] ) && $options['licensemela'] ? true : false );
		
		if ( isset( $options[ $this->tmc_api_key ] ) && ! empty( $options[ $this->tmc_api_key ] ) ) {
			$start = 3;
			$length = mb_strlen( $options[ $this->tmc_api_key ] ) - $start - 3;
			$mask_string = preg_replace( '/\S/', 'X', $options[ $this->tmc_api_key ] );
			$mask_string = mb_substr( $mask_string, $start, $length );
			$input_string = substr_replace( $options[ $this->tmc_api_key ], $mask_string, $start, $length );
		} else {
			$input_string = '';
		}
		if ( isset( $options[ $this->tmc_activation_email ] ) && ! empty( $options[ $this->tmc_activation_email ] ) ) {
			$start = 3;
			$length = mb_strlen( $options[ $this->tmc_activation_email ] ) - $start;
			$email_mask_string = preg_replace( '/\S/', 'X', $options[ $this->tmc_activation_email ] );
			$email_mask_string = mb_substr( $email_mask_string, $start, $length );
			$email_input_string = substr_replace( $options[ $this->tmc_activation_email ], $mask_string, $start, $length );
		} else {
			$email_input_string = '';
		}
		?>
		<form id="form_data" name="form" method="post">
			<input type="hidden" name="tmcore_license_form_submit" value="true" />
			<input type="hidden" name="tmcore_license_deactivate" value="true" />
			<?php wp_nonce_field( 'tmcore_license', 'tmcore_license_nonce' ); ?>
			<h2><?php echo __( 'TemplateMela License', 'templatemela-core' ); ?> <span class="tmc-license-status tmc-active"><?php echo __( 'Active', 'templatemela-core' ); ?></span></h2>
			<p class="activate-notice"><?php echo __( 'Your license is activated! Remote updates and theme support are enabled.', 'templatemela-core' ); ?></p>
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row"><?php echo 'Purchase Code'; ?></th>
						<td>
							<input id="api_key" name="<?php echo esc_attr( $this->tmc_data_key . '[' . $this->tmc_api_key . ']' ); ?>" disabled size="25" type="text" value="<?php echo esc_attr( $input_string ); ?>" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false">
						</td>
					</tr>
					<tr>
						<th scope="row"><?php echo 'License Email'; ?></th>
						<td>
							<input id="activation_email" name="<?php echo esc_attr( $this->tmc_data_key . '[' . $this->tmc_activation_email . ']' ); ?>" disabled size="25" type="text" value="<?php echo esc_attr( $email_input_string ); ?>">
						</td>
					</tr>
				</tbody>
			</table>
			<p class="submit">
				<?php if ( isset( $_GET['page'] ) && 'templatemela-setup' === $_GET['page'] ) { ?>
				<?php } else { ?>
					<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php echo esc_attr__( 'Deactivate', 'templatemela-core' ); ?>">
				<?php } ?>
			</p>
		</form>
		<?php
	}

	/**
	 * Sanitizes and validates all input and output for Dashboard
	 */
	public function validate_activation_options() {
		global $tmcore_license_interface_messages;
		// Load existing options, validate, and update with changes from input before returning.
		$options                                = $this->tmc_options;
		$form_data                              = ( isset( $_POST[ $this->tmc_data_key ] ) ? wp_unslash( $_POST[ $this->tmc_data_key ] ) : array() );
		$is_templatemela                        = ( isset( $form_data['licensemela'] ) ? trim( $form_data['licensemela'] ) : '' );
		$current_api_key                        = ( !empty( $options ) && isset( $options[ $this->tmc_api_key ] ) ? $options[ $this->tmc_api_key ] : '' );
		$options[ $this->tmc_api_key ]          = ( isset( $form_data[ $this->tmc_api_key ] ) ? trim( $form_data[ $this->tmc_api_key ] ) : '' );
		$options[ $this->tmc_activation_email ] = ( isset( $form_data[ $this->tmc_activation_email ] ) ? trim( $form_data[ $this->tmc_activation_email ] ) : '' );
		
		$api_email                              = ( isset( $form_data[ $this->tmc_activation_email ] ) ? trim( $form_data[ $this->tmc_activation_email ] ) : '' );
		$api_key                                = ( isset( $form_data[ $this->tmc_api_key ] ) ? trim( $form_data[ $this->tmc_api_key ] ) : '' );
		$activation_status                      = $this->get_setting_option( $this->tmc_activated_key );
		$checkbox_status                        = $this->get_setting_option( $this->tmc_deactivate_checkbox );
		$clear_options = array(
			$this->tmc_api_key          => '',
			$this->tmc_activation_email => '',
		);
		$settings = $options;
		if ( $is_templatemela && 'on' === $is_templatemela ) {
			$settings['licensemela'] = true;
		} else {
			$settings['licensemela'] = false;
		}
		if ( ( isset( $options['licensemela'] ) && $options['licensemela'] !== $settings['licensemela'] ) || ( ! isset( $options['licensemela'] ) && $settings['licensemela'] ) ) {
			$this->update_setting_option( $this->tmc_data_key, $settings );
			return;
		}
		if ( $activation_status == 'Deactivated' || $activation_status == '' || $api_key == '' || $api_email == '' || $checkbox_status == 'on' || $current_api_key != $api_key ) {
			if ( isset( $current_api_key ) && ! empty( $current_api_key ) ) {
				if ( $current_api_key != $api_key ) {
					$this->replace_license_key( $current_api_key );
				}
			}

			$args = array(
				'email'          => $api_email,
				'license_key'    => $api_key,
				'version'        => TMCORE_VERSION,
			);
			if ( $settings['licensemela'] ) {
				$activate_results = json_decode( $this->activate( $args ), true );
			} else {
				$activate_results = json_decode( $this->activate_envamonster( $args ), true ); // Envato / Monster
			}
			
			if ( isset( $activate_results['activated'] ) && $activate_results['activated'] === true ) {
				$tmcore_license_interface_messages[] = array(
					'type' => 'updated',
					'text' => __( 'TemplateMela license activated.', 'templatemela-core' ),
				);
				if ( isset( $activate_results['key'] ) && ! empty( $activate_results['key'] ) ) {
					$options['envamonster_key'] = $activate_results['key'];
				}
				$this->update_setting_option( $this->tmc_data_key, $options );
				$this->update_setting_option( $this->tmc_activated_key, 'Activated' );
			}

			if ( $activate_results == false ) {
				$tmcore_license_interface_messages[] = array(
					'type' => 'error',
					'text' => __( 'Connection failed to the License Key API server. Make sure your host servers php version has the curl module installed and enabled.', 'templatemela-core' ),
				);
				$this->update_setting_option( $this->tmc_data_key, $clear_options );
				$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
			}
			if ( isset( $activate_results['code'] ) ) {
				switch ( $activate_results['code'] ) {
					case '100':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
					case '101':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
					case '102':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
					case '103':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
					case '104':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
					case '105':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
					case '106':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $clear_options );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
						break;
				}

			}
		}

		return;
	}

	/**
	 * Deactivate the current license key before activating the new license key.
	 *
	 * @param string $current_api_key the license key.
	 */
	public function replace_license_key( $current_api_key ) {
		global $tmcore_license_interface_messages;
		$args = array(
			'email'       => $this->tmc_options[ $this->tmc_activation_email ],
			'license_key' => $current_api_key,
		);

		$reset = $this->deactivate( $args ); // reset license key activation.

		if ( true == $reset ) {
			return true;
		}
		$tmcore_license_interface_messages[] = array(
			'type' => 'error',
			'text' => __( 'The license could not be deactivated. Please try again.', 'templatemela-core' ),
		);

		return false;
	}

	/**
	 * Deactivates the license key to allow key to be used on another blog.
	 */
	public function validate_deactivation_options() {
		global $tmcore_license_interface_messages;
		$options            = $this->tmc_options;
		$activation_status  = $this->get_setting_option( $this->tmc_activated_key );
		$form_data          = ( isset( $_POST[ $this->tmc_data_key ] ) ? wp_unslash( $_POST[ $this->tmc_data_key ] ) : array() );
		$is_templatemela    = ( isset( $options['licensemela'] ) && $options['licensemela'] ? true : false );
		
		if ( ! is_array( $this->tmc_options ) ) {
			$this->tmc_options = array(
				$this->tmc_api_key          => '',
				$this->tmc_activation_email => '',
			);
		}
		$args = array(
			'email'            => ( isset( $this->tmc_options[ $this->tmc_activation_email ] ) ? $this->tmc_options[ $this->tmc_activation_email ] : '' ),
			'license_key'      => ( isset( $this->tmc_options[ $this->tmc_api_key ] ) ? $this->tmc_options[ $this->tmc_api_key ] : '' ),
			'version'          => TMCORE_VERSION,
			'envamonster_key'  => ( isset( $this->tmc_options['envamonster_key'] ) ? $this->tmc_options['envamonster_key'] : '' ),
		);
		$update_cleared = array(
			$this->tmc_api_key           => '',
			$this->tmc_activation_email  => '',
		);
		if ( $activation_status != 'Activated' ) {
			$this->update_setting_option( $this->tmc_instance_key, wp_generate_password( 12, false ) );
			$this->update_setting_option( $this->tmc_data_key, $update_cleared );
		}

		if ( $activation_status === 'Activated' && $this->tmc_options[ $this->tmc_api_key ] != '' && $this->tmc_options[ $this->tmc_activation_email ] != '' ) {
			if ( $is_templatemela ) {
				$activate_results = json_decode( $this->deactivate( $args ), true );
			} else {
				
				$activate_results = json_decode( $this->deactivate_envamonster( $args ), true );
			}

			if ( isset( $activate_results['deactivated'] ) && $activate_results['deactivated'] == true ) {
				if ( ! empty( $this->tmc_activated_key ) ) {
					$this->update_setting_option( $this->tmc_data_key, $update_cleared );
					$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					$tmcore_license_interface_messages[] = array(
						'type' => 'updated',
						'text' => __( 'TemplateMela license deactivated. ', 'templatemela-core' ),
					);
				}

				return;
			}

			if ( isset( $activate_results['code'] ) ) {

				switch ( $activate_results['code'] ) {
					case '100':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
					case '101':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
					case '102':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
					case '103':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
					case '104':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
					case '105':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
					case '106':
						$additional_info = ! empty( $activate_results['additional info'] ) ? esc_attr( $activate_results['additional info'] ) : '';
						$tmcore_license_interface_messages[] = array(
							'type' => 'error',
							'text' => "{$activate_results['error']}. {$additional_info}",
						);
						$this->update_setting_option( $this->tmc_data_key, $update_cleared );
						$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
					break;
				}

			}
		} elseif ( 'Activated' === $activation_status && ( '' === $this->tmc_options[ $this->tmc_api_key ] || '' === $this->tmc_options[ $this->tmc_activation_email ] ) ) {
			$update = array(
				$this->tmc_api_key => '',
				$this->tmc_activation_email => ''
			);

			$merge_options = array_merge( $this->tmc_options, $update );
			if ( ! empty( $this->tmc_activated_key ) ) {
				$this->update_setting_option( $this->tmc_data_key, $merge_options );
				$this->update_setting_option( $this->tmc_activated_key, 'Deactivated' );
				$tmcore_license_interface_messages[] = array(
					'type' => 'updated',
					'text' => __( 'TemplateMela license deactivated. ', 'templatemela-core' ),
				);
			}

			return;
		}
		return;
	}

	/**
	 * Loads admin style sheets and scripts
	 */
	public function scripts() {
		if ( ! isset( $_GET['page'] ) || 'avanam' !== $_GET['page'] ) {
			return;
		}
		wp_enqueue_script( 'templatemela-dashboard', TMCORE_URL . 'includes/addons/build/dashboard.js', array( 'wp-i18n', 'wp-element', 'wp-plugins', 'wp-components', 'wp-api', 'wp-hooks', 'wp-edit-post', 'lodash', 'wp-block-library', 'wp-block-editor', 'wp-editor', 'jquery' ), TMCORE_VERSION, true );
		wp_localize_script(
			'templatemela-dashboard',
			'baseProDashboardParams',
			array(
				'adminURL'   => admin_url(),
				'settings'   => get_option( 'templatemela_core_module_config' ),
				'activated'  => $this->get_setting_option( $this->tmc_activated_key ),
				'ajax_url'   => admin_url( 'admin-ajax.php' ),
				'ajax_nonce' => wp_create_nonce( 'base-ajax-verification' ),
			)
		);
		if ( function_exists( 'wp_set_script_translations' ) ) {
			wp_set_script_translations( 'templatemela-dashboard', 'templatemela-core' );
		}
	}

	/**
	 * Create Software API URL
	 *
	 * @param array $args for the url.
	 */
	public function create_software_api_url( $args ) {

		$api_url = add_query_arg( $args, $this->upgrade_url );

		return $api_url;
	}

	/**
	 * Activate the domain
	 *
	 * @param array $args for the activation.
	 */
	public function activate_envamonster( $args ) {
		// Envato Monster Add.
		if ( is_callable( 'network_home_url' ) ) {
			$site_url = network_home_url( '', 'http' );
		} else {
			$site_url = get_bloginfo( 'url' );
		}
		$site_url = preg_replace( '/^https/', 'http', $site_url );
		$site_url = preg_replace( '|/$|', '', $site_url );		

		$default_query = array(
			'wp'            => $GLOBALS['wp_version'],
			'site'          => $site_url,
			'timestamp'     => time(),
			'auth_version'  => '2',
		);
		$data = array(
			'auth_token'       => $this->get_password_hash( $args['email'], $args['license_key'], $site_url ),
			'email'            => $args['email'],
			'license_key'      => $args['license_key'],
			'license_domain'   => $this->tmc_domain,
			'platform'         => $this->tmc_domain,
			'request'          => 'activation',
			'instance'         => $this->tmc_instance_id,
			'product_id'       => $this->tmc_product_id,
			'product_name'     => $this->tmc_product_name,
			'software_version' => $args['version'],
			'site_url'         => $site_url,
			'packages'         => array(
				'details' => array(
					'ver'    => ( isset( $args['version'] ) ? $args['version'] : '' ),
					'key'    => ( isset( $args['envamonster_key'] ) ? $args['envamonster_key'] : '' ),
					'active' => true,
				),
			),
		);
		if ( isset( $data['auth_token'] ) ) {
			$data['iterations'] = 8;
		}

		$post_data = array(
			'request' => json_encode( $data ),
		);
		$remote_post_args = array(
			'timeout' => 20,
			'body'    => $post_data,
		);
		$request = '/?' . http_build_query( $default_query, '', '&' );
		$response = wp_remote_post(  untrailingslashit( $this->envamonster_api_url ) . $request, $remote_post_args );
		 
		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) != 200 ) {
			return false;
		}
		$response = json_decode( wp_remote_retrieve_body( $response ), true );

		$success = array();
		$data_key = '';
		if (isset($response['packages'])) {
			foreach ( $response['packages'] as $package => $data ) {
				if ( ! empty( $data['key'] ) && 'active' === $data['status'] ) {
					$success[] = 'activated';
					$data_key = $data['key'];
				}
			}
		}
		if ( ! empty( $success ) ) {
			$response['packages']  = true;
			$response['key']       =  $data_key;

			return json_encode($response);
		} else {
			return json_encode($response);
		}
	}

	/**
	 * Activate the domain
	 *
	 * @param array $args for the activation.
	 */
	public function deactivate_envamonster( $args ) {
		// Envato Monster Add.
		if ( is_callable( 'network_home_url' ) ) {
			$site_url = network_home_url( '', 'http' );
		} else {
			$site_url = get_bloginfo( 'url' );
		}
		$site_url = preg_replace( '/^https/', 'http', $site_url );
		$site_url = preg_replace( '|/$|', '', $site_url );		

		$default_query = array(
			'wp'           => $GLOBALS['wp_version'],
			'site'         => $site_url,
			'timestamp'    => time(),
			'auth_version' => '2',
		);
		$data = array(
			'auth_token'       => $this->get_password_hash( $args['email'], $args['license_key'], $site_url ),
			'email'            => $args['email'],
			'license_key'      => $args['license_key'],
			'license_domain'   => $this->tmc_domain,
			'platform'         => $this->tmc_domain,
			'request'          => 'deactivation',
			'instance'         => $this->tmc_instance_id,
			'product_id'       => $this->tmc_product_id,
			'product_name'     => $this->tmc_product_name,
			'software_version' => $args['version'],
			'site_url'         => $site_url,
			'packages'         => array(
				'details' => array(
					'ver'    => ( isset( $args['version'] ) ? $args['version'] : '' ),
					'key'    => ( isset( $args['envamonster_key'] ) ? $args['envamonster_key'] : '' ),
					'active' => true,
				),
			),
		);

		if ( isset( $data['auth_token'] ) ) {
			$data['iterations'] = 8;
		}
		$post_data = array(
			'request' => json_encode( $data ),
		);
		$remote_post_args = array(
			'timeout' => 20,
			'body'    => $post_data,
		);

		$request = '/?' . http_build_query( $default_query, '', '&' );
		$response = wp_remote_post(  untrailingslashit( $this->envamonster_api_url ) . $request, $remote_post_args );
		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) != 200 ) {
			return false;
		}
		$response = json_decode( wp_remote_retrieve_body( $response ), true );

		$success = array();
		$data_key = '';
		if (isset($response['packages'])) {
			foreach ( $response['packages'] as $package => $data ) {
				if ( isset( $data['status'] ) && 'deactive' === $data['status'] ) {
					$success[] = 'deactivated';
				}
			}
		}
		if ( ! empty( $success ) ) {
			$response['packages'] = true;
			return json_encode($response);
		} else {
			return json_encode($response);
		}
	}

	/**
	 * Checks if the software is activated or deactivated
	 *
	 * @param array $args the product args.
	 */
	public function status_envamonster( $args ) {
		// Envato Monster Add.
		if ( is_callable( 'network_home_url' ) ) {
			$site_url = network_home_url( '', 'http' );
		} else {
			$site_url = get_bloginfo( 'url' );
		}
		$site_url = preg_replace( '/^https/', 'http', $site_url );
		$site_url = preg_replace( '|/$|', '', $site_url );		

		$default_query = array(
			'wp'           => $GLOBALS['wp_version'],
			'site'         => $site_url,
			'timestamp'    => time(),
			'auth_version' => '2',
		);
		$data = array(
			'auth_token'       => $this->get_password_hash( $args['email'], $args['license_key'], $site_url ),
			'email'            => $args['email'],
			'license_key'      => $args['license_key'],
			'license_domain'   => $this->tmc_domain,
			'platform'         => $this->tmc_domain,
			'request'          => 'status',
			'instance'         => $this->tmc_instance_id,
			'product_id'       => $this->tmc_product_id,
			'product_name'     => $this->tmc_product_name,
			'software_version' => $args['version'],
			'site_url'         => $site_url,
			'packages'         => array(
				'details' => array(
					'ver'     => ( isset( $args['version'] ) ? $args['version'] : '' ),
					'key'     => ( isset( $args['envamonster_key'] ) ? $args['envamonster_key'] : '' ),
					'active'  => true,
				),
			),
		);

		if ( isset( $data['auth_token'] ) ) {
			$data['iterations'] = 8;
		}
		$post_data = array(
			'request' => json_encode( $data ),
		);
		$remote_post_args = array(
			'timeout' => 20,
			'body'    => $post_data,
		);

		$request = '/?' . http_build_query( $default_query, '', '&' );
		$response = wp_remote_post(  untrailingslashit( $this->envamonster_api_url ) . $request, $remote_post_args );
		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) != 200 ) {
			return false;
		}
		$response = json_decode( wp_remote_retrieve_body( $response ), true );
		
		$success = array();

		if ( isset( $response['status_check'] ) ) {
			$success[] = $response['status_check'];
		}

		if ( ! empty( $success ) ) {
			$response['packages'] = true;
			return json_encode($response);
		} else {
			return json_encode($response);
		}
	}

	public function get_password_hash( $username, $password, $site_url ) {
		$password = $this->get_pbkdf2( $password, $username );
		$salted_password = $password . $username . $site_url . $GLOBALS['wp_version'];
		$salted_password = substr( $salted_password, 0, max( strlen( $password ), 512 ) );
		$auth_token = wp_hash_password( $salted_password );
		return $auth_token;
	}
	public function get_salt( $username ) {
		return strtolower( trim( $username ) ) . 'wdHVwU&HcYcWnllo%kTUUnxpScy4%ICM29';
	}
	public function get_pbkdf2( $password, $username ) {
		
		return $this->pbkdf2(
			'sha256', 
			$password, 
			$this->get_salt( $username ), 
			131072, 
			64 / 2, 
			false
		);
		
	}
	
	//-----------------------------------------------------------------------------        
	
	/*
		* PBKDF2 key derivation function as defined by RSA's PKCS #5: https://www.ietf.org/rfc/rfc2898.txt
		* $algorithm - The hash algorithm to use. Recommended: SHA256
		* $password - The password.
		* $salt - A salt that is unique to the password.
		* $count - Iteration count. Higher is better, but slower. Recommended: At least 1000.
		* $key_length - The length of the derived key in bytes.
		* $raw_output - If true, the key is returned in raw binary format. Hex encoded otherwise.
		* Returns: A $key_length-byte key derived from the password and salt.
		*
		* Test vectors can be found here: https://www.ietf.org/rfc/rfc6070.txt
		*
		* This implementation of PBKDF2 was originally created by https://defuse.ca
		* With improvements by http://www.variations-of-shadow.com
		*/
	private function pbkdf2( $algorithm, $password, $salt, $count, $key_length, $raw_output = false ) {
		
		$algorithm = strtolower($algorithm);
		
		if(!in_array($algorithm, hash_algos(), true))
			trigger_error('PBKDF2 ERROR: Invalid hash algorithm.', E_USER_ERROR);
		
		if($count <= 0 || $key_length <= 0)
			trigger_error('PBKDF2 ERROR: Invalid parameters.', E_USER_ERROR);
	
	
		$hash_length = strlen(hash($algorithm, '', true));
		$block_count = ceil($key_length / $hash_length);
	
		$output = '';
		
		for($i = 1; $i <= $block_count; $i++) 
		{
			
			// $i encoded as 4 bytes, big endian.
			$last = $salt . pack("N", $i);
			
			// first iteration
			$last = $xorsum = hash_hmac($algorithm, $last, $password, true);
			
			// perform the other $count - 1 iterations
			for ($j = 1; $j < $count; $j++) 
			{
				$xorsum ^= ($last = hash_hmac($algorithm, $last, $password, true));
			}
			
			$output .= $xorsum;
			
		}
	
		if($raw_output)
			return substr($output, 0, $key_length);
		else
			return bin2hex(substr($output, 0, $key_length));
			
	}

	/**
	 * Activate the domain
	 *
	 * @param array $args for the activation.
	 */
	public function activate( $args ) {
		
		$defaults = array(
			'wc-api'           => 'am-software-api',
			'request'          => 'activation',
			'product_id'       => $this->tmc_product_id,
			'instance'         => $this->tmc_instance_id,
			'platform'         => $this->tmc_domain,
			'software_version' => $this->version,
		);
		$args = wp_parse_args( $defaults, $args );

		$target_url = esc_url_raw( $this->create_software_api_url( $args ) );

		$request = wp_safe_remote_get( $target_url, array( 'sslverify' => false ) );
		if ( is_wp_error( $request ) ) {
			// Lets try api address.
			$new_target_url = esc_url_raw( add_query_arg( $args, $this->fallback_api_url ) );
			$request        = wp_safe_remote_get( $new_target_url, array( 'sslverify' => false ) );
			if ( is_wp_error( $request ) || 200 !== wp_remote_retrieve_response_code( $request ) ) {
				return false;
			}
		} elseif ( 200 !== wp_remote_retrieve_response_code( $request ) ) {

			return false;
		}
		$response = wp_remote_retrieve_body( $request );

		return $response;
	}
	/**
	 * Deactivate the domain
	 *
	 * @param array $args for the deactivation.
	 */
	public function deactivate( $args ) {

		$defaults = array(
			'wc-api'     => 'am-software-api',
			'request'    => 'deactivation',
			'product_id' => $this->tmc_product_id,
			'instance'   => $this->tmc_instance_id,
			'platform'   => $this->tmc_domain,
		);

		$args = wp_parse_args( $defaults, $args );

		$target_url = esc_url_raw( $this->create_software_api_url( $args ) );

		$request = wp_safe_remote_get( $target_url, array( 'sslverify'  => false ) );
		if ( is_wp_error( $request ) ) {
			// Lets try api address.
			$new_target_url = esc_url_raw( add_query_arg( $args, $this->fallback_api_url ) );
			$request        = wp_safe_remote_get( $new_target_url, array( 'sslverify' => false ) );
			if ( is_wp_error( $request ) || 200 !== wp_remote_retrieve_response_code( $request ) ) {
				return false;
			}
		} elseif ( 200 !== wp_remote_retrieve_response_code( $request ) ) {
			return false;
		}
		$response = wp_remote_retrieve_body( $request );

		return $response;
	}

	/**
	 * Checks if the software is activated or deactivated
	 *
	 * @param array $args for the status check.
	 */
	public function status( $args ) {
		
		$defaults = array(
			'wc-api'     => 'am-software-api',
			'request'    => 'status',
			'product_id' => $this->tmc_product_id,
			'instance'   => $this->tmc_instance_id,
			'platform'   => $this->tmc_domain,
		);

		$args = wp_parse_args( $defaults, $args );

		$target_url = esc_url_raw( $this->create_software_api_url( $args ) );

		$request = wp_safe_remote_get( $target_url, array( 'sslverify'  => false ) );

		if ( is_wp_error( $request ) ) {
			// Lets try api address.
			$new_target_url = esc_url_raw( add_query_arg( $args, $this->fallback_api_url ) );
			$request        = wp_safe_remote_get( $new_target_url, array( 'sslverify' => false ) );
			if ( is_wp_error( $request ) || 200 !== wp_remote_retrieve_response_code( $request ) ) {
				return 'failed';
			}
		} elseif ( 200 !== wp_remote_retrieve_response_code( $request ) ) {
			return 'failed';
		}

		$response = wp_remote_retrieve_body( $request );

		return $response;
	}


	/**
	 * Get registered TGMPA plugins
	 *
	 * @return    array
	 */
	protected function get_tgmpa_plugins() {
		$plugins = array(
			'all'      => array(), // Meaning: all plugins which still have open actions.
			'install'  => array(),
			'update'   => array(),
			'activate' => array(),
		);

		foreach ( $this->tgmpa->plugins as $slug => $plugin ) {
			if ( $plugin['source_type'] != 'external' ) {
				continue;
			} else {
				$plugins['all'][ $slug ] = $plugin;
				if ( ! $this->tgmpa->is_plugin_installed( $slug ) ) {
					$plugins['install'][ $slug ] = $plugin;
				} else {
					if ( false !== $this->tgmpa->does_plugin_have_update( $slug ) ) {
						$plugins['update'][ $slug ] = $plugin;
					}
					if ( $this->tgmpa->can_plugin_activate( $slug ) ) {
						$plugins['activate'][ $slug ] = $plugin;
					}
				}
			}
		}

		return $plugins;
	}

}

if ( defined( 'TMTHEME_ID' ) && defined( 'TMTHEME_NAME' ) ) {
	global $tmcore_get_api_instance;
	$tmcore_get_api_instance = TemplateMela_Core_API_Manager::instance( TMTHEME_ID.'_activation', TMTHEME_ID.'_api_manager_instance', TMTHEME_ID.'_api_manager_activated', TMTHEME_ID, TMTHEME_NAME );
}